// (C) 2004 D. Gauchard <deyv@free.fr>
// Licensed under the terms of the GNU Public License version 2

#include "stdio.h"
#include "stdlib.h"
#include "string.h"
#include "ctype.h"
#include "errno.h"
#include "sys/stat.h"

#define LEN	1024
#define BIGLEN	8192

#if defined(__CYGWIN32) || defined(_WIN32)
#define snprintf(a,b,c...) sprintf(a,c)
#endif

void help (char* name)
{
	printf("Syntax: %s <file-name> <struct-name> <filename> [<filename>...]\n", name);
}

void build_defname (char* defname, char* filename)
{
	int i;
	char c;
	strcpy(defname, filename);
	i = 0;
	do
	{
		switch (c = filename[i])
		{
			case '.':
			case '-':
			case '/':
				c = '_';
				break;
		}
		defname[i++] = c;
	} while (c);
}
		
void build_defnamelen (char* defnamelen, char* filename)
{
	int i;
	char c;
	strcpy(defnamelen, filename);
	i = 0;
	do
	{
		switch (c = toupper(filename[i]))
		{
			case '.':
			case '-':
			case '/':
				c = '_';
				break;
		}
		defnamelen[i++] = c;
	} while (c);
	strcat(defnamelen, "_LEN");
}
		
int main (int argc, char* argv[])
{
	FILE* f;
	FILE* fc;
	FILE* fh;
	char fcname[LEN], fhname[LEN], temp[LEN], decl[BIGLEN];
	char* filename;
	char* headername;
	off_t size, i;
	char* buf;
	int nf;
	
	if (argc < 2)
	{
		help(argv[0]);
		return 1;
	}
	
	filename = argv[1];
	
	snprintf(fhname, LEN, "%s.h", filename);
	
	/*
	 * fopen failing in update means the file may not exist.
	 * Write file banner in that case.
	 */
	if ((fh = fopen(fhname, "r+")) == NULL)
	{
		if (errno != ENOENT)
		{
			perror(fhname);
			return 1;
		}
		if ((fh = fopen(fhname, "w")) == NULL)
		{
			perror(fhname);
			return 1;
		}
		fprintf(fh, "/* This file is automatically generated */\n"
			    "\n"
			    "#ifndef __BINWARE_H__\n"
			    "#define __BINWARE_H__\n"
			    "\n"
			    "#ifndef __BINWARE_STRUCT__\n"
			    "#define __BINWARE_STRUCT__\n"
			    "\n"
			    "typedef struct\n"
			    "{\n"
			    "\tconst char*\t\tname;\n"
			    "\tint\t\t\tsize;\n"
			    "\tconst unsigned char*\tdata;\n"
			    "} binware_s;\n"
			    "\n"
			    "#endif // __BINWARE_STRUCT__\n"
			    "\n");
	}
	else if (fseek(fh, 0, SEEK_END) == -1)
	{
		perror(fhname);
		return 1;
	}
	
	if (argc == 2)
	{
		fprintf(fh, "\n#endif // __BINWARE_H__\n");
		fclose(fh);
		return 0;
	}
	
	headername = argv[2];
	fprintf(fh, "extern binware_s binware_%s [];\n", headername);
	fclose(fh);

	snprintf(decl, BIGLEN, "binware_s binware_%s [] =\n{", headername);

	snprintf(fcname, LEN, "%s.c", filename);
	if ((fc = fopen(fcname, "r+")) == NULL)
	{
		if (errno != ENOENT)
		{
			perror(fcname);
			return 1;
		}
		if ((fc = fopen(fcname, "w")) == NULL)
		{
			perror(fcname);
			return 1;
		}
		fprintf(fc, "/* This file is automatically generated */\n"
			    "\n"
			    "#include <stdlib.h>\n"
			    "#include \"%s\"\n"
			    "\n",
			    fhname);
	}
	else if (fseek(fc, 0, SEEK_END) == -1)
	{
		perror(fcname);
		return 1;
	}

	for (nf = 3; nf < argc; nf++)
	{
		int ret;
		int decl_only;
		struct stat st;
		char defname[LEN];
		char defnamelen[LEN];
		char* binname = argv[nf];
		char* guilty;

		/*
		   do not produce binaries if binname contains "./"
		*/
		if ((decl_only = !!(guilty = strstr(binname, "./"))))
			memmove(guilty, &guilty[2], strlen(guilty) - 1);

		build_defname(defname, binname);
		build_defnamelen(defnamelen, binname);

		if (!decl_only)
		{
	
			if (stat(binname, &st) == -1)
			{
				perror("stat");
				return 1;
			}
			size = st.st_size;

			if ((f = fopen(binname, "rb")) == NULL)
			{
				perror(binname);
				return 1;
			}

			if ((buf = (char*)malloc(size)) == NULL)
			{
				perror("malloc");
				fprintf(stderr, "Cannot allocate %i bytes for file %s\n", (int)size, binname);
				return 1;
			}
	
			if ((ret = fread(buf, size, 1, f)) != 1)
			{
				if (ferror(f))
					perror(binname);
				fprintf(stderr, "Could not fully read file %s\n", binname);
				return 1;
			}
	
			fclose(f);
	
			fprintf(fc, "#define %s %i\n", defnamelen, (int)size);
	
			fprintf(fc, "unsigned char %s[%s] =\n{\t", defname, defnamelen);
			for (i = 0; i < size; i++)
			{
				fprintf(fc, "0x%02x, ", (unsigned char)buf[i]);
				if ((i != (size - 1)) && !((i+1) % 16))
					fprintf(fc, "\n\t");
			}
			fprintf(fc, "\n};\n\n");
		
			free(buf);
		}
		
		snprintf(temp, LEN, "\t{ \"%s\", %s, %s },\n", binname, defnamelen, defname);
		strcat(decl, temp);
	}
	fprintf(fc, decl);
	fprintf(fc, "\t{ NULL, 0, NULL }\n};\n\n");

	fclose(fc);

	return 0;
}
